;************procedure Mask version 1_0***************************
;Procedure	Mask
;Function	Check the input file type is correct
;         	In case of multiple dark files determine the minimum dark current per pixel
;        	Create mask of pixels that show a dark current above the specified cut off level
;
;Options	   Mask,/AUTO - Mask a fraction of pixels rather than a fixed dark current per pixel
;
;Input	   Raw data dark files series of raw data files (r1234567.fit)
;Output 	   Maskfile (m1234567.fit) containing mask image
;
;Reports to log	RUNNING PROCEDURE:Mask
; 	               Masking    #.##% of pixels with >  ##.## adu/hour
; 	               New mask image created:m1234567 using # files
; 	               Saving copy of Mask.gif
;Report files	   Copy of mask image (mask.gif)
;       	         Table of fraction of masked pixels versus cut off level (mask.txt)

;Error   WARNING: no dark file specified
;message FATAL ERROR: incorrect file type, file type is xxx
;      	ERROR: mask cut greater than specified maximum level
;
;Procedures called;
;read_control - Returns control data in structure c
;file_debias  - Reads pipeline header (c,h,f) and data array
;write_file   - Writes pipeline header (c,h,f) and data array to fits file
;printl       - Prints text message to log file
;
;***********************Procedure***********************************
pro Mask,AUTO=auto
;***************Read control data in structure c********************
read_control,c
;*******cancel operation if zero mask files are specified***********
if c.nmask eq 0 then print,'No dark file specified, no mask created'
if c.nmask eq 0 then return
printl,c.log,'RUNNING PROCEDURE:Mask
;******read header and debiased image of first dark file in list****
file_debias,c.maskfile,c,h,f,image
;***************Check file type*************************************
if not strcmp(strtrim(h.obstype,2),'DARK')  then $
printl,c.log,'FATAL ERROR: incorrect file type, file type is '+h.obstype
if not strcmp(strtrim(h.obstype,2),'DARK') then return
;*******************Get mask array***********************************
;*******define arrays to hold stack of dark images and output********
siz = size(image)
imagestack = fltarr(siz(1),siz(2),c.nmask)
;*************load flat images into stack array**********************
for i = 0,c.nmask-1 do begin
;*********read header and debiased dark image************************
   file_debias,c.dmask(i),c,h,f,image
;*********scale input to give dark current per hour******************
   imagestack(*,*,i) = image(*,*)*3600.0/float(h.darktime)
endfor
;*********Get minimum of dark current from stack array***************
if c.nmask gt 1 then image = min(imagestack,DIMENSION=3) else $
   image = imagestack(*,*,0)
;***********Get histogram of fraction of pixels vs dark current******
;********define dark current levels in pixel count per hour**********
level = findgen(c.maskmax)+1.0
;********define array to hold histogram values***********************
hist  = fltarr(c.maskmax)
for i = 0,c.maskmax-1 do begin
   index = where(image gt level(i),count)
   if count gt 0 then hist(i) = float(count)/float(n_elements(image))
endfor
;******Set maximum dark current for auto mode (mask,/auto)**************
;****set variable maskcut to give specified fraction of masked pixels***
index=where(hist gt c.masklev,count)
if count gt 0 then maskcut = level(count-1) else maskcut = max(level)
;****error if cut off level required for auto mode is too high**********
if maskcut ge c.maskmax and keyword_set(auto) then $
   printl,c.log,'ERROR: mask cut greater than specified maximum level'
;******Set maximum dark current for standard mode **********************
;****set variable maskcut to specified value****************************
if not keyword_set(auto) then maskcut = c.maskcut
;*****************Generate mask image (1/0)*****************************
siz = size(image)
mask = intarr(siz(1),siz(2))+1
index = where(image gt maskcut,count)
if count gt 0 then mask[index] = 0
;*********get percentage of masked pixels*******************************
maskpc = 100*float(count)/float(siz(1))/float(siz(2))
;*******save mask mask to mask file (m1234567.fit)**********************
write_file,c.pdir+c.pint+'m'+strtrim(c.maskfile,2),c,h,f,mask
;*******print summary information to screen and log file****************
printl,c.log,'Masking '+string(maskpc,FORMAT='(f7.2)')+$
    '% of pixels with >'+string(maskcut,FORMAT='(f7.2)')+' adu/hour'
printl,c.log,'New mask image created:m'+strtrim(c.maskfile,2)+$
    ' using '+ strtrim(c.nmask,2)+' files'
;***********save table of histogram values to log file******************
openw,nw,c.pdir+c.plog+'Mask.txt', /get_lun
printf,nw,'maskcut fraction'
for i = 0,c.maskmax-1 do printf,nw,level(i),hist(i),format='(I8,f8.5)'
close,nw   &  free_lun,nw
;**************log/log plot on histogram on screen**********************
window,1,xsize=400,ysize=300  & !p.multi=0   & !p.charsize=1
plot,level,hist,background=c.white,color=c.black,/XLOG,/YLOG,$
   xtitle='Cut off level (adu/hr)',ytitle='Fraction pixels masked'
;*************Display TV image of bias file image************************
;**********get constant to scale image to screen************************
tvsize = max([1+ siz(1)/fix(0.7*c.psize),1+ siz(2)/fix(0.7*c.psize)])
window,0,xsize=siz(1)/tvsize,ysize=siz(2)/tvsize
;******display TV image on screen***************************************
tv,rebin(mask(0:tvsize*(siz(1)/tvsize)-1,0:tvsize*(siz(2)/tvsize)-1)$
      ,siz(1)/tvsize,siz(2)/tvsize)*255
;******save copy of screen image to log file mask.gif*******************
printl,c.log,'Saving copy of Mask.gif'
WRITE_gif,c.pdir+c.plog+'Mask.gif',TVRD()
end
