;***************Procedure Circ version 1_0*************************
;Procedure	Circ
;Function	Check input file type and binning are correct
; 	         Trace centreline of active fibers across raw data image
; 	         Creates an offset image of the Circ file
; 	         Set values of relative fiber gains in file header
;
;Options	   Circ,/AUTO identifies fibers located in gap and overwrites values in control file
;
;Input	   Raw data file for a uniformly illuminated image with fibers in circle set up (r1234567.fit)
;Output 	   Circfile containing offset image and centerline data (c1234567.fit)
;
;Report to log	RUNNING PROCEDURE:Circ
; 	            ### lines identified with # fiber(s) in central gap, number(s) ## and ##
; 	            Fiber gains set to 1 / calculated from mean line counts / read from file xxx
; 	            New circ file created: c1234567.fit
;Report files  Table of pixel positions of peaks in image profile for each active fiber
;
;Error	WARNING :no circ file specified - using first flat file
;message FATAL ERROR: incorrect file type, file type is xxx
; 	      FATAL ERROR: incorrect width of CCD line profile, c.lwide/c.xbin must be even
;        FATAL ERROR: cannot select single fiber if Circfile is also a target file';
; 	      ERROR: saturation of circ image (>99% full scale)
; 	      WARNING: high circ image, ## of full scale
; 	      WARNING: low circ image, ## of full scale
; 	      FATAL ERROR: error in header data, incorrect fibers identified with gap
; 	      FATAL ERROR: file fiber_attn.txt not found
; 	      FATAL ERROR: list of fiber gains does not hold values for all valid fibers
;
;Procedures called;
;read_control - Returns control data in structure c
;read_data    - Reads pipeline header (c,h,f) and binary image of raw data file
;file_debias  - Reads pipeline header (c,h,f) and debiased image from data file
;circ_make    - Trace pixel positions of line centres on CCD image
;write_file   - Writes pipeline header (c,h,f) and data array to fits file
;write_control- Write control data to current control file (control_data.txt)
;printl       - Prints text message to log file
;
;*****************Procedure*****************************************
Pro Circ,auto=AUTO
;***************Read control data in structure c********************
read_control,c
printl,c.log,'RUNNING PROCEDURE:Circ
;**************clear open windows***********************************
while (!d.window ge 0) do wdelete,!d.window
;********Check if dedicated circ file is being used*****************
if c.ncirc eq 0 then printl,c.log,$
      'WARNING: no circ file specified - using first flat file'
;******read header and debiased image of first circ file in list****
file_debias,c.circfile,c,h,f,image
;***************check file type*************************************
if (not strcmp(strtrim(h.obstype,2),'FLAT')) and $
   (not strcmp(strtrim(h.obstype,2),'TARGET')) then printl,c.log,$
    'FATAL ERROR: incorrect file type, file type is '+h.obstype
if (not strcmp(strtrim(h.obstype,2),'FLAT')) and $
   (not strcmp(strtrim(h.obstype,2),'TARGET')) then stop
;****check parameter c.lwide is compatible with x binning***********
if (c.lwide/c.xbin mod 2) gt 0 then printl,c.log,'FATAL ERROR:'+$
   ' incorrect width of CCD line profile, c.lwide/c.xbin must be even'
if (c.lwide/c.xbin mod 2) gt 0 then stop
;*************create median circ image (if required)*****************
if c.ncirc gt 1 then begin
;*****define arrays to hold stack of circ images *******************
   scale = fltarr(c.ncirc)
   siz = size(image)
   inarr = fltarr(siz(1),siz(2),c.ncirc)
   for i = 0,c.ncirc-1 do begin
;*********read header and debiased circ image***********************
      file_debias,c.dcirc(i),c,h,f,image
;*******scale image to its mean value before taking median *********
      scale(i) = 1.0/mean(image)
      inarr(*,*,i) = image*scale(i)
   endfor
;****define image as median of stack array and rescale**************
      image = median(inarr,DIMENSION=3)/mean(scale)
endif
;****check specified circ file is OK to use for case singe fiber****
index = where(strtrim(c.dstar,2) eq strtrim(c.circfile,2),count)
if count gt 0 and c.fiberid gt 0 and c.fiberid lt c.nfiber then printl,c.log,$
  'FATAL ERROR: cannot select single fiber if Circfile is also a target file'
if count gt 0 and c.fiberid gt 0 and c.fiberid lt c.nfiber then stop
;**************check circ signal level******************************
;*****measure peak level in (ystep) cuts through the flat image*****
;*****cuts averaged over several pixels in wavelength direction*****
ystep = fix(c.ymax/c.ybin/fix(c.circlev+1))
pklev = fltarr(c.circlev)
for i = 0,c.circlev - 1 do pklev(i) = max(median(dimension=2,$
         image(*,ystep*(i+1)-c.yavg/c.ybin:ystep*(i+1)+c.yavg/c.ybin)))
;******express peak value as a fraction of full scale***************
pklev = pklev/(2.0*float(c.ccdzero))
;**Error if normalised peak is under/over specified value***********
if max(pklev) ge 0.99 then printl,c.log,$
  'ERROR: saturation of circ image (>99% full scale)'
if max(pklev) ge c.sighi then printl,c.log,'WARNING: high circ image, '$
+string(max(pklev),format='(f5.2)')+' of full scale'
if max(pklev) le c.siglo then printl,c.log,'WARNING: low circ image, '$
+string(min(pklev),format='(f5.2)')+' of full scale'
;**************mask edges of centre section*****************v*******
;**Option only used in case of a bad alignment of traces ************
;***where a line falls partially in gap at centre of ccd ************
if c.circcutr gt 0 then for i = 0,c.circcutr/c.xbin-1 do $
   image(c.oscan3/c.xbin+i,*)=image(c.oscan3/c.xbin,*)
if c.circcutl gt 0 then for i = 0,c.circcutl/c.xbin-1 do $
   image(c.oscan2/c.xbin-i,*)=image(c.oscan2/c.xbin,*)
;*************************identify line centres *********************
;***************Calling circ_make*************************************
;Input:  median circfile header and image (c,h,f,image)
;Output: ypix - array of y pixel values used for image cuts
;        peak - array of x-pixel value of line centres for image cuts
;        num  - Number of lines identified in circ image for each cut
;        gapfiber - Array identifying fibers in gap (max of 4)
circ_make,c,h,f,image,ypix,peak,num,gapfiber
;*****Check if fibers in gap agrees with header data********************
if not keyword_set(auto) then begin
   dfiber = [c.dfiber1,c.dfiber2,c.dfiber3,c.dfiber4]
   if mean(abs(dfiber(sort(dfiber))-gapfiber(sort(gapfiber)))) gt 0 then $
   printl,c.log,'ERROR: header data shows incorrect fiber numbers in CCD gap'
endif
;*****if using auto mode (circ,/auto) then overwrite control file ******
;*****parameters c.dfober# with new numbers for fibers in gap***********
if keyword_set(auto) then begin
   index = where(gapfiber gt 0,ngap)
   c.dfiber1 = gapfiber(0)     &      c.dfiber2 = gapfiber(1)
   c.dfiber3 = gapfiber(2)     &      c.dfiber4 = gapfiber(3)
   write_control,c
endif
;*******print summary information to screen and log file****************
;*****showing number of lines and fibers in gap*************************
index = where(gapfiber gt 0,ngap)
dummy = strtrim(num(0),2)+' lines identified with '$
+ strtrim(ngap,2)+' fiber(s) in gap, number(s) '
for i = 0,ngap-1 do dummy = dummy +', '+ strtrim(gapfiber(i),2)
printl,c.log,dummy
;***********save table of line centre data to file circ.txt**************
openw,nw,c.pdir+c.plog+'Circ.txt',/get_lun
printf,nw,'#Fiber Line  Pixel numbers for centres'
for i = 0,c.nfiber-1 do if (not strcmp(f(i).type,'X')) and f(i).line gt 0 $
   then printf,nw,i+1,f(i).line,peak(f(i).line-1,*),format='(I4,I6,7F9.2)'
close,nw      &    free_lun,nw
nline = num(0)
;*********creat array to hold line centre data at all y values***********
offset  = fltarr((c.lwide/c.xbin+4)*nline,c.ymax/c.ybin)
yind = indgen(c.ymax/c.ybin)
;***interpolate x pixel values of line centres as function of y pixel*****
;***save results to designated columns of array offset********************
for i = 0,nline-1 do offset(i*(c.lwide/c.xbin+4),*)=spline(ypix,peak(i,*),yind)
;***************Calling file_offset*************************************
;Input:  circfile header and image (c,h,f,image)
;        array offset of x pixel values of line centres as function of y
;Output: array output - contining offset data (see user manual) plus a
;        copy of the original circ image shifted along the x-axis
;        so that the centres of each line fall at a constant x value
file_offset,c,h,image,offset,output
;
;***********add value of fiber gains to header data (parameter f.gain)***
;***********Evaluate approximate offset of lines in the x direction******
;*used to estimate fiber gains for option c.fibrattn=1 (see user manual)*
yoffo  = fltarr(c.nfiber)
j = 0
for i = 0,c.nfiber-1 do begin
;******get nominal fiber position counter across CCD image***************
   if  i mod (c.nfiber/c.nfiducl) gt 0 then j = j + 1
;*********translate this to a nominal distance from the centerline*******
   yoffo(i) = (2*j - c.nfiber+c.nfiducl)/float(c.nfiber-c.nfiducl)
;*******calculate the yoffset as a function of this distance squared*****
;*****using maximum offset across CCD from header dispersion********
   yoffo(i) = yoffo(i) * yoffo(i) * fix(c.fiberdip/float(h.disperse*c.ybin))
;*******add a term to take account or the 1/2/3 stagger of fibers********
   yoffo(i) = fix(yoffo(i) + (j mod 3) * c.fiberoff/c.ybin)
endfor
yoffo(where(f.stat eq 2)) = 0
;********If fibrattn=0 then f.gain=1*************************************
f(*).gain = 1.0
if c.fibrattn eq 0 then printl,c.log,'Fiber gains set to 1.0'
;********If fibrattn=1 then calculate fiber gains by averaging traces of**
;***circ file over fixed line width and length corrected for y offsets****
if c.fibrattn eq 1 then begin
   for i = 0,c.nfiber-1 do if f(i).line gt 0 and f(i).type ne 'X' $
     then  f(i).gain = total(output((f(i).line-1)*(c.lwide/c.xbin+4)+2:$
     f(i).line*(c.lwide/c.xbin+4)-2,yoffo(i):c.ymax/c.ybin+yoffo(i)-max(yoffo)-1))
   index = where(f.gain gt 0.0)
   f.gain = f.gain/mean(f(index).gain)
   printl,c.log,'Fiber gains calculated from mean line counts'
endif
;***If fibrattn=2 then f.gain is read from file  in standards directory****
;***If fibrattn=3 then f.gain is read from file  in project log directory**
if c.fibrattn eq 2 or c.fibrattn eq 3 then begin
   file  = 'control_standards'+path_sep()+'fiber_attn.txt'
   if  c.fibrattn eq 3 then file = c.pdir+c.plog+'fiber_attn.txt'
   printl,c.log,'Fiber gains read from file; '+file
   f(*).gain = 0
;******open specified file and read fiber attenuations f.gain***************
   openr,nr,file,/get_lun,ERROR=err
   if err ne 0 then printl,c.log,'FATAL ERROR: file fiber_attn.txt not found'
   if err ne 0 then stop
   dummy = 'dummy'
   readf,nr,dummy
   while not eof(nr) do begin
      readf,nr,i,gg
      f(i-1).gain = gg
   endwhile
   close,nr     & free_lun,nr
;*****************check all active fibers have a value for gain***************
   j = 0
   for i = 0,c.nfiber-1 do if f(i).line gt 0 and  f(i).gain eq 0 then j=j+1
      if j gt 0 then printl,c.log,'FATAL ERROR: list of fiber gains'+$
      ' does not hold values for all active fibers'
      if j gt 0 then stop
endif
;******save image containing line centre data to circ file (c1234567.fit)*****
write_file,c.pdir+c.pint+'c'+strtrim(c.circfile,2),c,h,f,output
printl,c.log,'New circ file created:c'+strtrim(c.circfile,2)
end

