;****************procedure star_make version 1_0*************************
;Procedure	star_make,c,h,f,file,spectra,mag,err
;
;Function   Generate sky subtracted spectra on a common wavelength base
;           Scale spectra to flat profile (if specified in control file)
;
;Option     star_make,c,h,f,file,spectra,mag,err,/plot
;           saves individual fiber plots of spectra for each observation
;
;Input:     header data (c,h,f) and array of intermediate spectra (spectra)
;           file number (1234567) of starfile being processed
;Output:    arrays of magnitude (mag) and error (err) of sky subtracted
;           spectra on a common wavelength base (see section 2.3 of developer notes
;           row 0 wavelength, row 1 to 160 fiber spectra, row 161 median sky
;
;Report to log   Reading offset sky from o1234567  (in case of offset sky)
;                Average sky fiber signal  ##.# +/- ##.#  e-/pixel
;
;Error   FATAL ERROR: wavelength calibration data not present
;message FATAL ERROR: offset sky file o1234567 not found (if offset sky specified)
;        WARNING: sky signal is low for scaled sky subtraction
;        WARNING: Parameter c.skymax is low, less than 3 sigma of median sky signal.'
;
;Procedures called;
;read_file    - Reads pipeline header (c,h,f) and image of pipeline fits file
;sky_sub      - Subtract sky spectrum from a single fiber target spectrum
;flat_scale   - Get flat lamp profile for  a single fiber
;printl       - Prints text message to log file
;
;************************Procedure********************************************
pro star_make,c,h,f,file,spectra,mag,err,PLOT=plot
;******set startype=1 for starfile otherwise zero*****************************
index = where(c.dstar eq file,startype)
;*****set up arrays for output spectra****************************************
mag = fltarr(c.nfiber+2,c.nwl)
err = fltarr(c.nfiber+2,c.nwl)
;************count number targets of each type (sky,star or circle set up)****
index = where(f.type eq 'S' and f.stat eq 1,nsky)
index = where(f.type eq 'P' and f.stat eq 1,nstar)
index = where(f.type eq 'C' and f.stat eq 1,ncirc)
;************Check wavelength calibration data present************************
for i = 0,c.nfiber-1 do if f(i).type ne 'X' and  f(i).line gt 0 $
     then if mean(spectra(8*(f(i).line-1),*)) eq 0.0 then begin
   printl,c.log,'FATAL ERROR: wavelength calibration data not present'
   stop
endif
;**Set common wavelength scale - log scale type(0) or linear scale type (1)***
if c.wltype  eq 0 then wl = exp(alog(c.wlmin) $
     + indgen(c.nwl)*(alog(c.wlmax)-alog(c.wlmin))/double(c.nwl-1))
if c.wltype  eq 1 then wl = c.wlmin + (c.wlmax - c.wlmin)*$
     indgen(c.nwl)/double(c.nwl-1)
;*******fill column 1 of output spectra arrays with wavelength data***********
mag(0,*) = wl
err(0,*) = wl
;*****set up arrays for sky spectra*******************************************
skyspec = fltarr(c.nfiber,c.nwl)
skyerr  = fltarr(c.nfiber,c.nwl)
;********************Read in median offset sky *******************************
;**only do if current file is a star type and offset sky set in control file**
index = where(c.dstar eq file,istar)
if c.skyoset eq 1 and istar gt 0 then begin
;***********check offset sky file exists**************************************
   osetfile = c.pdir+c.pspec+'o'+strtrim(c.osetfile,2)
   openr,nr,osetfile+'.fit',/get_lun,ERROR=error
   if error ne 0 then begin
       printl,c.log,'FATAL ERROR: offset sky file o'+osetfile+' not found'
       stop
   endif
   close,nr    & free_lun,nr
;**********read offset sky file***********************************************
   printl,c.log,'Reading offset sky from '+'o'+strtrim(c.osetfile,2)
   read_file,osetfile,cs,hs,fs,magoset,page=0
   read_file,osetfile,cs,hs,fs,erroset,page=1
;***********Copy offset sky spectra to sky arrays scaling for exposure time***
   if c.fiberid eq 0 or c.fiberid ge c.nfiber then begin
       for i = 0,c.nfiber-1 do skyspec(i,*) = magoset(i+1,*)*h.exptime/hs.exptime
       for i = 0,c.nfiber-1 do skyerr(i,*)  = erroset(i+1,*)*h.exptime/hs.exptime
    endif else begin
;***********treat case of single fiber extraction*****************************
      for i = 0,c.nfiber-1 do skyspec(i,*) = magoset(1,*)*h.exptime/hs.exptime
      for i = 0,c.nfiber-1 do skyerr(i,*)  = erroset(1,*)*h.exptime/hs.exptime
    endelse
endif
;**********************Evaluate median sky spectrum***************************
;*****only do if no. sky targets >0 and offset sky not set in control file****
if nsky gt 0 and c.skyoset eq 0 then begin
;***********set up arrays to stack all sky spectra****************************
   skymed = fltarr(c.nfiber,c.nwl)
   skyvar = fltarr(c.nfiber,c.nwl)
   scale  = fltarr(c.nfiber)
   nsky   = 0
;******Interpolate intermediate sky spectra onto a common wavelegth base*******
;******save results sequentially to stack arrays (skymed and skyvar)**********
   for i = 0,c.nfiber-1 do if f(i).type eq 'S' and f(i).line gt 0 then begin
      skymed(nsky,*)=SPLINE(spectra(8*(f(i).line-1),*),$
                  spectra(8*(f(i).line-1)+5,*),wl,/double)
      skyvar(nsky,*)=SPLINE(spectra(8*(f(i).line-1),*),$
                       spectra(8*(f(i).line-1)+6,*),wl,/double)
;********scale sky spectra by fiber attenuation and normalised to mean********
      skymed(nsky,*) =skymed(nsky,*)/f(i).gain
      skyvar(nsky,*) =skyvar(nsky,*)/f(i).gain^2
      scale(nsky) = mean(skymed(nsky,*))
      skymed(nsky,*)= skymed(nsky,*)/scale(nsky)
      skyvar(nsky,*)= skyvar(nsky,*)/scale(nsky)/scale(nsky)
;*************increment counter of number of sky spectra**********************
      nsky = nsky + 1
   endif
;********truncate stack arrays to contain only sky spectra********************
   skymed = skymed(0:nsky-1,*)*mean(scale(0:nsky-1))
   skyvar = skyvar(0:nsky-1,*)*mean(scale(0:nsky-1))*mean(scale(0:nsky-1))
;*********evaluate median sky and error (root of variance)********************
   skyspec(0,*) = median(skymed,dimension=1)
   skyerr(0,*)  = sqrt(abs(median(skyvar,dimension=1)/float(nsky)))
;*********copy median sky to all sky spectra**********************************
   for i = 0,c.nfiber-1 do skyspec(i,*) = skyspec(0,*)
   for i = 0,c.nfiber-1 do skyerr(i,*)  = skyerr(0,*)
;*************print message to log********************************************
   printl,c.log,'Average sky fiber signal '+$
       string(mean(skyspec),format='(f7.1)')+'+/-'+$
       string(stddev(skyspec),format='(f7.1)')+' e-/pixel'
endif
;*******************output median sky spectra***********************************
if nsky gt 0  and c.skyoset eq 0 then begin
;******set RH column of output spectra arrays equal to median sky spectra******
   mag(c.nfiber+1,*)  = skyspec(0,*)
   err(c.nfiber+1,*)  = skyerr(0,*)
;**************plot median sky spectra*****************************************
   tit = ' File '+strtrim(file,2)+'_FSKY'+'   Median of '+strtrim(nsky,2)+' fibers'
   if mean(skyspec) gt 20 then ymax = 10*mean(skyspec) else ymax = 200
   plot,mag(0,*),mag(c.nfiber+1,*),background=-1,color=c.black,xstyle=1,title=tit,$
   xtitle = 'Wavelength (A)', ytitle ='e/pixel',yrange=[0,ymax]
   oplot,mag(0,*),err(c.nfiber+1,*),color=c.red
   if keyword_set(plot) then $
     WRITE_gif,c.pdir+c.ptemp+strtrim(file,2)+'_FSKY'+'.gif',TVRD()
endif
;**********Evaluate output spectra for star (or offset sky) targets************
;***check level of median sky spectra compatible with type of sky subtraction**
if c.skytype gt 1 and stddev(mag(c.nfiber+1,*))/2.0/float(c.ccdzero) lt $
       c.skymin then printl,c.log,$
      'WARNING: sky signal is low for scaled sky subtraction'
;*****check cut off level for sky lines is >> typical sky signal level***
if c.skytype eq 3 and c.skymax lt 3*stddev(skyspec(0,*)) then printl,c.log,$
'WARNING: Parameter c.skymax is low, less than 3 sigma of median sky signal.'
;**********loop through all valid targets********************************
for i = 0,c.nfiber-1 do if f(i).type ne 'X' and f(i).line gt 0 then begin
;****interpolate intermediate spectra over masked sections****************
;********set first and last point of column holding mask data to 1********
   spectra(8*(f(i).line-1)+7,0) = 1
   spectra(8*(f(i).line-1)+7,c.ymax/c.ybin-1) = 1
;Set arrays to hold  pixel No., magnitude and variance of unmasked points
   tempw = dblarr(c.ymax/c.ybin)
   temps = dblarr(c.ymax/c.ybin)
   tempe = dblarr(c.ymax/c.ybin)
 ;**initialise counter of number of pixels containing unmasked point*******
   j=0
 ;***fill arrays containing pixel No., mag and val for unmasked point******
   for m = 0,c.ymax/c.ybin-1 do begin
      tempw(j) = spectra(8*(f(i).line-1),m)
      temps(j) = spectra(8*(f(i).line-1)+5,m)
      tempe(j) = spectra(8*(f(i).line-1)+6,m)
 ;***increment counter if original spectra contains unmasked point**********
      if spectra(8*(f(i).line-1)+7,m) gt 0.5 then j = j + 1
   endfor
 ;*truncate arrays of wavelength, mag and var at max No. of unmasked points*
   tempw = tempw(0:j-1)
   temps = temps(0:j-1)
   tempe = tempe(0:j-1)
;*****interpolate data for unmasked points over all wavelength values*******
;**skip interpolation in case of arc lamp spectra - a non standard action***
   if not strpos(h.obstype,'ARC') ge 0 then begin
      speco=SPLINE(tempw,temps,spectra(8*(f(i).line-1),*),/double)
      errro=SPLINE(tempw,tempe,spectra(8*(f(i).line-1),*),/double)
   endif else begin
      speco =  spectra(8*(f(i).line-1)+1,*)
      errro =  spectra(8*(f(i).line-1)+2,*)
   endelse
;*******Interpolate intermediate spectra onto a common wavelength base******
   starspec = SPLINE(spectra(8*(f(i).line-1),*),speco ,wl,/double)
;******taking sqrt of variance to give rms error***************************
   starerr  = sqrt(abs(SPLINE(spectra(8*(f(i).line-1),*),errro ,wl,/double)))
   ;starmask = SPLINE(spectra(8*(f(i).line-1),*),$
   ;                     spectra(8*(f(i).line-1)+7,*),wl,/double)
   starmask =wl*0.0+ 1.0
;**************************Remove bad data values**************************
;***********cut spectra at maximum credible value**************************
   specmax = float((2*c.ccdzero-1)*(c.lwide/c.xbin/2))
   index = where(abs(starspec) gt specmax,count)
   index = where(abs(starerr) gt specmax,count)
   ;***************scale spectrum for fiber gain******************************
   starspec = starspec/f(i).gain
   starerr  = starerr /f(i).gain
;******Subtract median sky from star spectrum (but not offset sky)*********
;************************Calling sky_sub***********************************
;Input:  fiber number counter (i) and header data (c,f)
;        common wavelength scale (wl), star spectrum and its error
;        sky spectrum and its error
;Output: sky subtracted star spectrum and error
;        type of sky subtraction used set by parameter c.skytype
   if f(i).type eq 'P' and startype eq 1 then $
      sky_sub,i,c,f,wl,starspec,starerr,skyspec(i,*),skyerr(i,*)
 ;******set error value negative over masked section of source spectra*****
   index = where(starmask lt 0.95,count)
   if count gt 1 then starerr(index) = - starerr(index)
;***************Calling flat_scale*****************************************
;Input:  control data (c), wavelength and magnitude of intermediate flat spectrum
;        common wavelength scale (wl)
;Output: flat spectrum profile i.e. smoothed version of flat spectrum
;         normalised to its average value on common wavelength scale
;         unit value returned if parameter c.flatscale=0
   flat_scale,c,spectra(8*(f(i).line-1),*),spectra(8*(f(i).line-1)+3,*)$
           ,wl,flat_scale
;***********Scale spectra to smoothed flat spectrum*************************
   mag(i+1,*)  = float(starspec*flat_scale)
   err(i+1,*)  = float(starerr *flat_scale)
;**********************plot target spectrum*********************************
   tit = strtrim(file,2)+'_'+f(i).fib+'  type '+f(i).type+'  SNR '+$
    strtrim(fix(mean(mag(i+1,*))/mean(abs(err(i+1,*)))),2)+'    '+f(i).name
   if mean(mag(i+1,*)) gt 70 then ymax = 2*mean(mag(i+1,*)) else ymax = 200
   plot,wl,mag(i+1,*),background=-1,color=c.black,xstyle=1,title=tit,$
     xtitle = 'Wavelength (A)', ytitle ='electrons/pixel',yrange=[0,ymax]
   oplot,wl,err(i+1,*),color=c.red
;***if option /plot then save plot to file 1234567Msky.gif in /temp**********
   if keyword_set(plot) then $
      WRITE_gif,c.pdir+c.ptemp+strtrim(file,2)+'_'+f(i).fib+'.gif',TVRD()
endif
;***write summary text file (p1234567.txt) see section 2.3 of developer notes
if startype eq 1 then begin
   openw, nw, c.pdir+c.pspec+'p'+strtrim(file,2)+'.txt',/get_lun
   printf,nw,'FILE','LINE','TYPE','MEAN','SNR','RA','DEC','MAGN','SKY_SUB','NAME'$
          ,FORMAT='(A4,A6,A5,A5,5X,A3,3X,A3,11X,A3,13X,A4,2X,A7,6X,4A)'
   if nsky gt 0 then begin
      ra = h.ra       &  dec = h.dec
      strput,ra ,' ',strpos(ra,   ':')    &  strput,dec,' ',strpos(dec,':')
      strput,ra ,' ',strpos(ra,   ':')    &  strput,dec,' ',strpos(dec,':')
      printf,nw,'FSKY',0,'S', mean(skyspec),mean(skyspec)/mean(skyerr),ra,dec,999$
      ,0.0,'Median Sky',FORMAT='(A4,3X,I3,A3,F9.1,F7.0,2X,A13,A13,3X,F6.2,F7.3,5X,A-25)'
   endif
   for i = 0,c.nfiber-1 do if not strcmp(f(i).type,'X')  and $
     f(i).line gt 0 then printf,nw,f(i).fib,f(i).line,f(i).type,mean(mag(i+1,*)),$
     mean(mag(i+1,*))/mean(abs(err(i+1,*))),f(i).ra,f(i).dec,f(i).magn,f(i).sky,$
     f(i).name,FORMAT='(A4,3X,I3,A3,F9.1,F7.0,2X,A13,A13,3X,F6.2,F7.3,5X,A-25)'
   close,nw     &      free_lun,nw
endif
end