;******************Procedure Star version 1_0************************
;Procedure	Star
; Function  Optimal extraction of spectra for a set of star files
;           Interpolates spectra onto a common wavelength scale
;           Sky subtraction using either median value of sky fibers or offset sky spectra
;
;Options	   Star,/NOEXTRACT - skips optimal extraction stage
;        	Star,/PLOT  - saves plots of individual spectra for each fiber
;           Star,/AUTO - Optimal extraction vs pixel number (i.e. no wavelength calibration)
;
;Input	   Raw data  images  of designated targets(r.1234567.fit)
;Output	   Starfiles (s1234567.fit) containing intermediate target spectra
;           Specfiles (p1234567.fit) sky subtracted spectra over common wavelength range
;           ASCII tables (p1234567.txt) fiber no, line, RA, Dec, Mag, Mean, SNR, sky lev,name
;
;Report to log	RUNNING PROCEDURE:Star
;              For file number(s), 1234567, 1234567, 1234567
;              Add wavelength calibration from file: a1234567
;              Star files created: s1234567 (repeat for each target)
;            	Spectra saved to  file p1234567 (repeat for each target)

;Error 	FATAL ERROR: readout binning does not match control data
;message FATAL ERROR: control file shows incompatible sky type
;        Cannot have skytype 0, and skyoset 0 for since fiber extraction
;
;Procedures called;
;read_control - Returns control data in structure c
;read_file    - Reads pipeline header (c,h,f) and image of pipeline fits file
;extr_file    - Optimal extract target spectra to produced intermediate spectra
;star_make    - Get sky subtracted spectra on a common wavelegth base
;write_file   - Writes pipeline header (c,h,f) and data array to fits file
;printl       - Prints text message to log file
;
;*******************Procedure***************************************
pro Star,PLOT=plot,NOEXTRACT=noextract,AUTO=auto
;***************Read control data in structure c**********************
read_control,c
printl,c.log,'RUNNING PROCEDURE:Star
;********print list of files to analyse to screen and log file********
dummy = 'For file number(s)'
for i = 0,c.nstar-1 do dummy  = dummy+', '+strtrim(c.dstar(i),2)
printl,c.log,dummy
;**********get intermediate spectra for specified star file(s)********
for k = 0,c.nstar-1 do if not keyword_set(noextract) then begin
;***************Calling extr_file*************************************
;Input:  starfile number
;        number of iterations for optimal extraction routine (c.staritn)
;Output: header information relating to starfile image (h,f)
;        array (spectra) containing intermediate spectra vs pixel number
   extr_file,c.dstar(k),c.staritn,h,f,spectra
;******add wavelength calibration data to file of intermediate spectra*
   if not keyword_set(auto) then begin
      siz = size(spectra)
;***make vector (type) containing object type for each line of image***
      type = strarr(siz(1)/8)
      for i = 0,c.nfiber-1 do if f(i).line gt 0 then type(f(i).line-1)=f(i).type
;****read the intermediate spectra array of the current arcfile********
      read_file,c.pdir+c.pint+'a'+strtrim(c.arcfile,2),ca,ha,fa,specarc
;copy wavelength calibration data to the intermediate star spectra array
      for i=0,siz(1)/8-1 do if type(i) ne 'X' then spectra(8*i,*) = specarc(8*i,*)
      printl,c.log,'Add wavelength calibration from file: a'+ strtrim(c.arcfile,2)
   endif
;**save intermediate spectra for each starfile to file (s1234567.fit)**
   write_file,c.pdir+c.pint+'s'+strtrim(c.dstar(k),2),c,h,f,spectra
   printl,c.log,'Star file created: s'+strtrim(c.dstar(k),2)
endif
;***********finish if running in auto mode (star,/auto)******************
;***else continue to subtract sky and recast on common wavelength range***
if keyword_set(auto) then return
;**************check Sky options are compatible with other settings******
if c.fiberid gt 0 and c.fiberid lt c.nfiber and c.skytype gt 0 and $
  c.skyoset eq 0 then begin
   printl,c.log,'FATAL ERROR: control file shows incompatible sky type'
   printl,c.log,'Cannot have skytype 0, and skyoset 0 for since fiber extraction'
   stop
endif
;*******************Set graphic window***********************************
window,xsize=c.psize,ysize=fix(c.psize/4)  & !P.CHARSIZE=1  & !p.multi=0
;***************Read control data in structure c*************************
read_control,c
;***get output spectra onto a common base for each star file*************
for k = 0,c.nstar-1 do begin
;*******Read header and intermediate spectra for a file******************
   read_file,c.pdir+c.pint+'s'+strtrim(c.dstar(k),2),cc,h,f,spectra
;***************get sky subtracted spectra on common wl base*************
   printl,c.log,'Evaluating spectra on common wavelength base'
;***************Calling star_make****************************************
;Input:  header data (c,h,f) and array of intermediate spectra (spectra)
;        file number (1234567) of starfile being processed
;Output: arrays of magnitude (mag) and error (err) of sky subtracted
;        spectra on a common wavelegth base (see section 2.3 of developer notes
;        row 0 wavelength, row 1 to 160 fiber spectra, row 161 median sky
   if keyword_set(plot) $
       then star_make,c,h,f,c.dstar(k),spectra,mag,err,/plot $
       else star_make,c,h,f,c.dstar(k),spectra,mag,err
;***save intermediate spectra for each starfile to file (s1234567.fit)***
   write_file,c.pdir+c.pint+'s'+strtrim(c.dstar(k),2),c,h,f,spectra
;*****Resize mag and err files in case of single target data*************
;*********row 0 wavelength, row 1 fiber spectrum, row 2 blank************
   if c.fiberid gt 0 and c.fiberid le c.nfiber then begin
      mag_fib  = mag(c.fiberid,*)   &      err_fib = err(c.fiberid,*)
      wl = mag(0,*)
      mag = fltarr(3,c.nwl)         &      err = fltarr(3,c.nwl)
      mag(0,*) = wl                 &      err(0,*) = wl
      mag(1,*) = mag_fib            &      err(1,*) = err_fib
   endif
;*Save output spectra to fits file with page 1 for mag and page 2 for err*
   write_file,c.pdir+c.pspec+'p'+strtrim(c.dstar(k),2),c,h,f,mag,page=0
   write_file,c.pdir+c.pspec+'p'+strtrim(c.dstar(k),2),c,h,f,err,page=1
   printl,c.log,'Spectra saved to file p'+strtrim(c.dstar(k),2)
endfor
end
