;******************procedure read_data version 1_0************************
;procedure read_data,nfile,c,h,f,image
;Function  read specified fits data file with AF2/WYFFOS header (r1234567.fit)
;          where the file directory path is specified by the control data
;Option    read_file,nfile,c,h,f,image,path=xxx reads file from path xxx
;
;Input:    Data file ile name including path but without .fit extension
;Output:   File header data (c,h,f), array of specified page (image)
;
;Module set fiber status in header data (f.stat) as follow;
;         1  Fiber OK to allocate to target
;         2  Fiducial fiber
;         3  Fiber flagged as dead in data file header
;         4  Fiber OK to allocate but trace is located in CCD gap so no line
;         5  Fiber OK but not analysd since pipeline operating in single target mode
;
;Module set fiber types in header data (f.type) as follows;
;         X  Unallocated fiber (parked, fiducial or dead)
;         P  Fiber allocated to star target
;         S  Fiber allocated to sky target (including all fibers named PLACE_XY)
;         C  Active fibers (status 1 or 4) positioned in circle set up
;
;Error   FATAL ERROR: file xxx not found
;message FATAL ERROR: image size does not match control data
;        WARNING: Data file header shows different number of dead fibers, ##, flagged
;
;Procedures called;
;read_control - Returns control data in structure c
;printl       - Prints text message to log file
;
;***********************Procedure*****************************************
pro read_data, nfile,c,h,f,image,PATH=path
;***************Read control data in structure c***************************
read_control,c
;***************check that the specified file exists***********************
if not keyword_set(PATH) then path = c.pdir+c.pdata
file = path +'r'+ strtrim(string(nfile),2) + '.fit'
openr,nr,file,/get_lun,ERROR=err
if err ne 0 then printl,c.log,'FATAL ERROR: file '+file+' not found'
if err ne 0 then stop
close,nr     & free_lun,nr
;**read fits file header (head) and page 0 (blank) array (image)***********
print,'reading file   '+file
image =mrdfits(file,0,head)
;******read filts file page 1 containing a the binnary data array (image)**
image =mrdfits(file,1)
;***************Check size of image compatible with control data***********
siz = size(image)
if siz(2) lt c.ymax/c.ybin then $
printl,c.log,'FATAL ERROR: image size does not match control data'
if siz(2) lt c.ymax/c.ybin then stop
;***********clip off lower (blue) mend of spectra (if required)***********
if c.yzero gt 0 then image = image[*,(c.yzero/c.ybin):siz(2)-1]
;*set maximum length of spectra (in pixels). This clips red end of spectra
image = image[*,0:(c.ymax/c.ybin-1)]
;****set structure for general header parameter (h)***********************
;****descriptors of header data (h) are section 1.2 of developer notes**
h = {h,run:0L,obstype:'x',wyffos:'x'$
    ,cenwave:0,disperse:0,linesmm:0,lamp:'x',wyfconf:'x'$
    ,wyfgrat:'x',wyforder:'x',confra:'x',confdec:'x',field:'x'$
    ,ndead:0,nfiber:0,nfiduc:0,ra:'x',dec:'x',equinox:'x'$
    ,jd:0.0D,start:'x',xbin:0,ybin:0,exptime:'x',darktime:0.0}
;***********read values of parameters (h) from fits file header***********
h.run     = sxpar(head,'RUN')
h.obstype = sxpar(head,'OBSTYPE')
h.wyffos  = sxpar(head,'INSTRUME')
h.cenwave = sxpar(head,'CENWAVE')
h.disperse= sxpar(head,'DISPERSI')
h.linesmm = sxpar(head,'LINESMM')
h.lamp    = sxpar(head,'WYFCLAMP')
h.wyfconf = sxpar(head,'WYFCONF')
h.wyfgrat = sxpar(head,'WYFGRAT')
h.wyforder= sxpar(head,'WYFORDER')
h.field   = sxpar(head,'AF2FLDNM')
h.confra  = sxpar(head,'AF2FLDRA')
h.confdec = sxpar(head,'AF2FLDDC')
h.nfiber  = sxpar(head,'AF2NFIB')
h.ndead   = sxpar(head,'AF2NDIS')
h.nfiduc  = sxpar(head,'AF2NFID')
h.ra      = sxpar(head,'RA')
h.dec     = sxpar(head,'DEC')
h.equinox = sxpar(head,'EQUINOX')
h.jd      = sxpar(head,'JD')
h.start   = sxpar(head,'STSTART')
h.xbin    = sxpar(head,'CCDXBIN')
h.ybin   =  sxpar(head,'CCDYBIN')
h.exptime = sxpar(head,'EXPTIME')
h.darktime= sxpar(head,'DARKTIME')
;***set structure for fiber specific header parameters (f)*********
;descriptors of header data (f) are in section 1.3 of developer notes
fiber = {fiber,stat:'x',line:0,fib:'x',type:'x',name:string(30)$
      ,magn:'x',ra:'x',dec:'x',gain:0.0,sky:0.0,x:0.0D,y:0.0D}
f = replicate ({fiber},c.nfiber)
;**define values of fiber parameters (f) from fits file header******
for i = 0,c.nfiber-1 do begin
;********define  fiber name (f.name)********************************
   if i+1 lt 10 then NAME = 'F00'+strtrim(i+1,2)
   if i+1 ge 10 and i lt 100 then NAME = 'F0'+strtrim(i+1,2)
   if i+1 ge 100 then NAME = 'F'+strtrim(i+1,2)
   f(i).fib   = NAME
;****read values of fiber parameters (f) from fits file header******
   f(i).stat  = strtrim(sxpar(head, NAME+'STAT'),2)
;read target type from fits file header setting type=X if none given
   f(i).type  = strtrim(sxpar(head, NAME+'TYPE'),2)
   if strlen(f(i).type) eq 0 then f(i).type = 'X'
;********read target name and magnitude from fits file header******
   f(i).name  = strtrim(sxpar(head, NAME+'NAME'),2)
   f(i).magn  = strtrim(sxpar(head, NAME+'MAGN'),2)
;****************read target RA and Dec from fits file header******
   f(i).ra    = strtrim(sxpar(head, NAME+'RA'),2)
   f(i).dec   = strtrim(sxpar(head, NAME+'DEC'),2)
;*********read x and y position of fiber on plate******************
   f(i).x     = strtrim(sxpar(head, NAME+'FIBX'),2)
   f(i).y     = strtrim(sxpar(head, NAME+'FIBY'),2)
;**set fiber gain to 1 and sky scaling parameter to zero***********
;*these parameters are generally reset later by pipeline procedures
   f(i).gain  = 1.0
   f(i).sky   = 0.0
;**designate fibers placed at plate  position x,y as sky fibers****
   if strpos(f(i).name,'PLACE_XY') ge 0 then f(i).type='S'
endfor
;****check No. fibers flagged dead (status 3) is same as fits header
index = where(f(*).stat eq 3,count)
if h.ndead ne count then printl,c.log,'WARNING: Data file header '+$
'shows different number of dead fibers, '+strtrim(count,2)+', flagged'
if h.ndead ne count then h.ndead = count
;**********step type of any dead fibers to X***********************
f(where (f.stat ne '1')).type = 'X'
;set status of stat=1 fibers flagged as gap fibers in control data to 4
gapfiber = [c.dfiber1,c.dfiber2,c.dfiber3,c.dfiber4]
for i = 0,4-1 do if gapfiber(i) gt 0 and gapfiber(i) lt c.nfiber $
   then if f(gapfiber(i)-1).stat eq 1 then f(gapfiber(i)-1).stat='4'
index = where(gapfiber gt 0,ngap)
index = intarr(c.nfiber)
;***get number lines on CCD image if all fibers used (i.e. circle set up)
j = h.nfiber - h.nfiduc - h.ndead -  ngap
;*allocate line numbers to fibers in reverse order across CCD image*
for i = 0,c.nfiber-1 do if f(i).stat eq 1 then begin
      f(i).line = strtrim(j,2)
      j = j - 1
endif
;Indentify fibers inside plate radius corresponding to circle set up
index = where(sqrt(f.x*f.x + f.y*f.y) lt c.rcircle and f.stat eq 1,nrcircle)
if nrcircle eq 0 then for i =0,c.nfiber-1 do $
;****if all active fibers are in circle set up then set fiber header data
;**set type as c, name as circle, RA and Dec as zero and magnitude as 999
    if f(i).stat eq 1 or f(i).stat eq 4 then begin
      f(i).type  = 'C'
      f(i).name = 'circle'           &    f(i).magn  = '999'
      f(i).ra   = '00 00 00.000'     &    f(i).dec   = '00 00 00.000'
  endif
;**********Set fiber parameter for case of single fiber extraction**************
if c.fiberID gt 0 and c.fiberID le c.nfiber then begin
;****check if file type is and arc,star or offset sky if it is then*************
;***** set fiber status of all active but non-selected fibers to type 5*********
   index = where(strtrim(c.darc ,2) eq strtrim(nfile,2),narc)
   index = where(strtrim(c.dstar,2) eq strtrim(nfile,2),nstar)
   index = where(strtrim(c.doset,2) eq strtrim(nfile,2),noset)
   if (nstar + narc + noset) gt 0 then  for i =0,c.nfiber-1 do $
    if (f(i).stat eq '1' or f(i).stat eq '4') and i ne (c.fiberID-1) then begin
      f(i).stat = '5'             &    f(i).type  = 'X'
      f(i).name = 'blank'         &    f(i).magn  = '999'
      f(i).ra   = '00 00 00.000'  &    f(i).dec   = '00 00 00.000'
   endif
endif
end
