;****************Procedure Offset version 1_0*************************
;Procedure	Offset
;Function	Optimal extraction of set of offset sky observations
;           Interpolates spectra onto a common wavelength scale
;           Evaluate median values of offset sky spectra from repeated frames
;
;Options	   Offset,/NOEXTRACT - skips optimal extraction stage
;
;Input	   Raw data images  of offset sky
;Output	   File(s) containing intermediate offset sky spectra(t1234567.fit)
;           File of median offset sky spectra over common wavelength range (o1234567.fit)
;
;Report to log	RUNNING PROCEDURE:Offset
;              For file number(s), 1234567, 1234567, 1234567
;              Add wavelength calibration from file: a1234567
;              Offset file created: t1234567
;              Evaluating median offset sky spectra for target fibers
;              Median offset sky spectra saved to  file o1234567
;              Saving plots to directory C:/CepOB3_data/temp/
;
;Error   FATAL ERROR: no offset sky files specified
;Message	FATAL ERROR: no target fibers found for sky offset
;
;Procedures called;
;read_control - Returns control data in structure c
;read_file    - Reads pipeline header (c,h,f) and image of pipeline fits file
;extr_file    - Optimal extract target spectra to produced intermediate spectra
;star_make    - Get spectra on a common wavelegth base
;medan_make   - Get median spectra on a common wavelegth base
;write_file   - Writes pipeline header (c,h,f) and data array to fits file
;printl       - Prints text message to log file
;
;**************************procedure***********************************
pro Offset,NOEXTRACT=noextract,NOPLOT=noplot
;***************Read control data in structure c***********************
read_control,c
printl,c.log,'RUNNING PROCEDURE:Offset
;**********Return if no offset sky called or necessary files missing***
if c.skyoset ne 1 then printl,c.log,'Sky offset not called for'
if c.skyoset ne 1 then return
if c.noset eq 0   then printl,c.log,'FATAL ERROR: no offset sky files specified'
if c.noset eq 0  then stop
;********print list of files to analyse to screen and log file********
dummy = 'For file numbers'
for i = 0,c.noset-1 do dummy  = dummy+', '+strtrim(c.doset(i),2)
printl,c.log,dummy
;**********get intermediate spectra for offset sky file***************
for k = 0,c.noset-1 do if not keyword_set(noextract) then begin
;***************Calling extr_file*************************************
;Input:  offset sky file number
;        number of iterations for optimal extraction routine (c.staritn)
;Output: header information relating to offset sky image (h,f)
;        array (spectra) containing intermediate spectra vs pixel number
   extr_file,c.doset(k),0,h,f,spectra
;******check file contains at least on valid target********************
   index = where(f.type eq 'P',count)
   if count eq 0 then printl,$
          clog,'FATAL ERROR: no target fibers found for sky offset'
   if count eq 0 then stop
;******name all valid targets (type P) as "Offset_sky"*****************
   f(index).name = 'Offset_sky'
;****designate all other fibers (i.e. not type P) as invalid (type X)**
   index = where(f.type ne 'P')
   f(index).type = 'X'
;***add wavelength calibration data to files of intermediate spectra***
   siz = size(spectra)
;***make vector (type) containing object type for each line of image***
   type = strarr(siz(1)/8)
   for i = 0,c.nfiber-1 do if f(i).line gt 0 then type(f(i).line-1)=f(i).type
;****read the intermediate spectra array of the current arcfile********
   read_file,c.pdir+c.pint+'a'+strtrim(c.arcfile,2),ca,ha,fa,specarc
;**copy wavelength calibration data to the offset sky spectra array****
   for i=0,siz(1)/8-1 do if type(i) ne 'X' then spectra(8*i,*) = specarc(8*i,*)
   printl,c.log,'Add wavelength calibration from file: a'+ strtrim(c.arcfile,2)
;*save intermediate spectra for each offset sky to file (t1234567.fit)*
   write_file,c.pdir+c.pint+'t'+strtrim(c.doset(k),2),c,h,f,spectra
   printl,c.log,'Offset file created: t'+strtrim(c.doset(k),2)
endif
;
;***Recast on common wavelength range and determine median spectra******
;*******************Set graphic window*********************************
window,xsize=1000,ysize=250  & !P.CHARSIZE=1  & !p.multi=0
;****set up arrays to hold stack of output spectra and median spectra**
if c.fiberid gt 0 and c.fiberid le c.nfiber then col = 3 else col = c.nfiber+2
mags  = fltarr(col,c.nwl,c.noset)
errs  = fltarr(col,c.nwl,c.noset)
magm  = fltarr(col,c.nwl)
errm  = fltarr(col,c.nwl)
;********initialise counter of total exposure time********************
exptime = 0
;***get output spectra onto a common base for each offset sky file****
for k = 0,c.noset-1 do begin
;****Read header and intermediate spectra for each starfile in turn***
   read_file,c.pdir+c.pint+'t'+strtrim(c.doset(k),2),cs,h,f,spectra
;*******set run number for header data of output file*****************
   if k eq 0 then run = h.run
;***************Calling star_make*************************************
;Input:  header data (c,h,f) and array of intermediate spectra (spectra)
;        file number (1234567) of offset sky file being processed
;Output: arrays of magnitude (mag) and error (err) of offset sky spectra
;        spectra on a common wavelength base (see section 2.3 of developer notes
;        row 0 wavelength, row 1 to 160 fiber spectra, row 161 blank
   star_make,c,h,f,c.doset(k),spectra,mag,err
;
;*****Resize mag and err files in case of single target data***********
;*********row 0 wavelength, row 1 fiber spectrum, row 2 blank**********
   if c.fiberid gt 0 and c.fiberid le c.nfiber then begin
      mag_fib  = mag(c.fiberid,*)   &      err_fib = err(c.fiberid,*)
      wl = mag(0,*)
      mag = fltarr(3,c.nwl)         &      err = fltarr(3,c.nwl)
      mag(0,*) = wl                 &      err(0,*) = wl
      mag(1,*) = mag_fib            &      err(1,*) =err_fib
   endif
;*****Copy output spectra for single offset sky observation************
   mags(*,*,k) = mag
   errs(*,*,k) = err
;***********increment total exposure time******************************
   exptime = exptime + h.exptime
endfor
;********set values of header data for output file*********************
h.run     = run
h.exptime = exptime
;**********Get median offset sky spectra for target fibers*************
printl,c.log,'Evaluating median offset sky spectra for target fibers'
for i = 0,c.nfiber-1 do if (f(i).type eq 'P')  then  begin
;*set column number in mags/errs arrays to be processed to get median
   if c.fiberid gt 0 and c.fiberid le c.nfiber then na = 1 else na = i+1
;***************Calling medan_make*************************************
;Input:  observation type 'o' used for offset sky observation
;Input:  Observation header data (h,f)
;        Stacked array of spectra on common wavelength base (mags & errs)
;        fiber number (i+1)
;        column number (na) of spectra for which median is found
;Output: array containing median magnitudes (magm) and errors (errm) of
;        offset sky spectra with designated column (na) filled
;        Screen shows plot of individual and median spectra for fiber i+1
   medan_make,'o',c,f,mags,errs,i+1,na,magm,errm
;********************save plot to file*********************************
   fib = f(i).fib
   strput,fib,'M',strpos(fib,'F')
;****Save plot of individual and median spectra for fiber number ###***
;****plot saved to file 1234567M###.gif in directory /temp*************
   WRITE_gif,c.pdir+c.ptemp+strtrim(c.osetfile,2)+'_'+fib+'.gif',TVRD()
endif
;Save output spectra to fits file with page 1 for mag and page 2 for err
write_file,c.pdir+c.pspec+'o'+strtrim(c.osetfile,2),c,h,f,magm,page=0
write_file,c.pdir+c.pspec+'o'+strtrim(c.osetfile,2),c,h,f,errm,page=1
;*****Messages to screen and logfile showing module actions done*******
printl,c.log,'Median offset sky spectra saved to file o'+strtrim(c.dstar(0),2)
printl,c.log,'Saving plots to directory '+c.pdir+c.ptemp
!p.multi=0
end
