;****************Procedure Atlas  version 1_0*************************
;Procedure	Atlas
;Function   GUI that is used to create and edit the arcline data for wavelength calibration
;           Select the line  the arc spectra used to calculate y-offsets
;           Select the  Atlas file name and common wavelength range for the arc spectra
;           Select  the wavelength and pixel position of lines used for wavelength calibration
;           Calibrate wavelength against pixel position for active fibers
;
;Input	   Intermediate arc spectra (a1234567.fit)
;           Contol_data.txt - file containing current control data
;Output	   Contol_data.txt - modified control file with changes made to arcline data
;
;Report to log	RUNNING PROCEDURE:Atlas
; 	            Overwriting arcline data for control file ### (if selected)
; 	            New arcline calibration data saved to control file (if selected)
; 	            Quit arcline calibration without changing control file (if selected)
;
;Error	FATAL ERROR: Atlas file: xxx not found
;message ERROR:  Atlas shows no lines in specified wavelength range'
;
;Procedures called;
;read_control - Returns control data in structure c
;read_file    - Reads pipeline header (c,h,f) and image of pipeline fits file
;extr_file    - Optimal extract target spectra to produced intermediate spectra
;arc_offset   - Get relative offset of individual arc spectra in pixels
;arc_atlas    - Plot overlay of arc spectra and plot atlas lines
;arc_cal      - Add wavelength calibration to data array of arc spectra
;arc_gui      - Graphical user interface to edit arcline data in control file
;write_control- Write control data to current control file (control_data.txt)
;printl       - Prints text message to log file
;

;*********************Procedure*****************************************
pro Atlas
;******Read control data structure c from current control file**********
read_control,c
printl,c.log,'RUNNING PROCEDURE:Atlas
;save copy of input control data to co to reset control data if Quit****
co = c
;****read the intermediate spectra array of the current arcfile*********
read_file,c.pdir+c.pint+'a'+strtrim(c.arcfile,2),ca,h,f,spectra
;*****Set flags in control data to initialise Atlas procedure***********
c.atlflag  = 0
c.atlaline = 0
;**If no atlas file is specified set name as header parameter h.lamp****
if strlen(strtrim(c.atlfile,2)) eq 0 then  c.atlfile = h.lamp
;******Overwrite current control file with data in structure c**********
write_control,c
;*****************Calibrate wavelength**********************************
;The loop below causes a GUI displaying the arcline data. In parallel
;procedures are called to generate graphical displays of arc spectra
;The case executed is set by control parameter c.atlflag -set in the GUI
;
;    Case 0:Select the arcline used to find the  y-offsets
;    Case 1:Select a atlas file, the wavelength range and arclines
;    Case 2:View wavelength solution using current arcline data
;    Case 3:Accept the calibration and the update control file
;    Case 9:Quit procedure (after resetting control data to old values)
;
;**variable j holds the number of calls to the loop (maximum 50)********
j = 0
while c.atlflag lt 5 and j le 50 do begin
   j = j+1
;******Read control data structure c from current control file**********
   read_control,c
   case c.atlflag of
;*****Case 0:Select the arcline used to find the  y-offsets*************
      0: begin
;***************Calling arc_offset**************************************
;Input:   arcfile header and intermediate spectra (c,h,f,spectra)
;Output:  yoff - relative offset of individual arc spectra in pixels found
;               by comparing position of a selected line in the spectra
         arc_offset,c,h,f,spectra,yoff,/window
      end
;*****Case 1:Select a atlas file, the wavelength range and arclines*****
      1: begin
;******Read control data structure c from current control file**********
         read_control,c
;;***************Calling arc_atlas***************************************
;Input:   arcfile header (c,h,f) and intermediate spectra (spectra)
;         yoff - relative offset of individual arc spectra in pixels (yoff)
;Function:Plot overlay of arc spectra offset by values in yoff
;         Plot Atlas lines in wavelength limits c.wlmin to c.wlmax
         arc_atlas,c,h,f,spectra,yoff
      end
;*****Case 2: View wavelength solution using current arcline data*******
      2: begin
;***************set plot window*****************************************
         window,xsize=fix(0.7*c.psize),ysize=fix(0.7*c.psize)
         !P.MULTI = [0,1,5,0,0]  & !P.CHARSIZE=2
;***************Calling arc_offset**************************************
;************see notes above********************************************
         arc_offset,c,h,f,spectra,yoff
;***************Calling arc_cal*****************************************
;Input:  arcfile header and intermediate arc spectra (c,h,f,spectra)
;        yoff - relative offset of individual arc spectra
;Output: wavelength calibration added data array of arc spectra (spectra)
;        using polynomial fit of wavelength vs pixel number
;        array of offsets between calibration data and polynomial fits (del)
;        limits of common wavelength range found from spectra (wlmin,wlmax)
         arc_cal,c,h,f,spectra,yoff,del,wlmin,wlmax
      end
;*****Case 3: Accept the calibration and the update control file********
      3: begin
;******Read control data structure c from current control file**********
         read_control,c
;******Update wavelength limits in control file if outside range********
         if c.wlmin lt wlmin or c.wlmax gt wlmax then begin
            c.wlmin = max([wlmin,c.wlmin])
            c.wlmax = min([wlmax,c.wlmax])
            write_control,c
         endif
;**If wavelength limits are changed write a note to screen and log file*
         if c.wlmin ne co.wlmin or c.wlmax ne co.wlmax then $
            printl,c.log,'New wavelength limits saved to control file'
         printl,c.log,'New arcline calibration data saved to control file'
;***set control parameter c.altflag=5 (showing calibration is accepted)***
         c.atlflag=5
      end
;*****Case 9: Quit procedure (after resetting control data to old values)
      9: begin
;*****save copy of initial control data held in co to control file*******
         write_control,co
         printl,c.log,'Quit arcline calibration without changing control file'
      end
   endcase
;******Calling arc_gui (if control parameter c.atlflag=0,1,2 or 3) *****
;Input:   current value of control data (c)
;Function:allows user to unpdate arcline data held current control file
   if c.atlflag lt 4 then arc_gui,c
;********if control parameter c.atlflag=9 then quit procedure************
   if c.atlflag gt 5 then return
endwhile
;**If calibration accepted save fit statistics to log file Atlas.txt******
openw,nw,c.pdir+c.plog+'Atlas.txt',/get_lun
;****get mean and rms fit of polynomial for each calibration line********
index = where(f.type ne 'X' and f.line gt 0,nfiber)
meandel = total(del,1)/float(nfiber)
rmsdel  = sqrt(total(del*del,1)/float(nfiber))
;*****print results to screen and to log file****************************
printf,nw,'#Arcline  wavelength  mean_delta_fit  rms_delta_fit'
print,'Arcline  wavelength  mean_delta_fit  rms_delta_fit'
j = 0
for i = 0,c.narcline-1 do if c.aflag(i) eq 1 then begin
   printf,nw,i+1,c.acalwl(i),meandel(j),rmsdel(j),format='(i6,f13.4,2f13.3)'
   print,i+1,c.acalwl(i),meandel(j),rmsdel(j),format='(i6,f13.4,2f13.3)'
   j = j+1
endif
close,nw  &  free_lun,nw
end